<?php

namespace Drupal\cookie_consent\Form;

use Drupal;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Configure Cookie consent settings for this site.
 */
class CookieConsentSettingsForm extends ConfigFormBase {

  private const settings = 'cookie_consent.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cookie_consent_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::settings,
    ];
  }

  public function defaultConfigValues() {
    return [

      'settings_dialog_label' => 'Privacy settings',
      'website_name' => $this->config('system.site')->get('name'),
      'accept_button_label' => 'Accept cookies',
      'deny_button_label' => 'Reject cookies',
      'settings_button_label' => 'Cookie settings',
      'reload_on_save' => 1,
      'enable_deny_button' => 1,
      'enable_close_button' => 0,
      'close_button_accept' => 1,
      'message' => "This website makes use of cookies and other tracking technologies to get a proper representation of the website and to provide you with as much user convenience as possible. For more information, click <a href='/privacy-statement' target='_blank'>here</a>",
      'functional_title' => 'Functional cookies',
      'functional_description' => 'Functional or technical cookies are cookies that are required to make the website work. These cookies facilitate a specific functionality of the website and improve your experience, for example by remembering your choices and preferences (for example a language choice) or by storing more complex decisions; with these cookies you can navigate our website and use its various functions.',
      'functional_default_state' => 1,
      'functional_togglable' => 0,

      'analytical_title' => 'Analytical cookies',
      'analytical_description' => 'These cookies enable us to recognise users and to track their numbers and to see how users navigate through our website. This helps us to improve navigation for our users and to make it easier for users to find what they are looking for. We only use the information obtained by means of these cookies to improve the performance of our website.',
      'analytical_default_state' => 1,
      'analytical_togglable' => 1,

      'tracking_title' => 'Tracking cookies',
      'tracking_description' => 'Tracking cookies, also known as marketing cookies or advertising cookies, are cookies that are used within a domain or across several domains to record the users’ surfing behaviour. Ultimately, they enable specific customised offers. The use of tracking cookies requires informed consent from you as the user. If you wish to accept these cookies, please use the button to agree by actively sliding the button from ‘off’ to ‘on’.',
      'tracking_disabled_message' => "Tracking cookies, also known as marketing or advertising cookies, record a users browsing behavior within a domain or across multiple domains. Your browser indicates that you have 'Do Not Track' enabled in your settings. As a result, we have disabled the use of tracking cookies.",
      'tracking_default_state' => 1,
      'tracking_togglable' => 1,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $settings = $this->config(static::settings);
    $form['gtm_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GTM code'),
      '#description' => $this->t('GTM code for this domain'),
      '#placeholder' => 'GTM-XXXXXXX',
      '#required' => TRUE,
      '#default_value' => $settings->get('gtm_code'),
    ];
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cookie Consent and Google Tag manager are enabled'),
      '#required' => FALSE,
      '#default_value' => $settings->get('enabled'),
    ];

    $form['global_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Global settings'),
    ];
    $form['global_settings']['website_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Website name'),
      '#description' => $this->t('Name shown in the settings dialog.'),
      '#required' => TRUE,
      '#default_value' => $this->getSettingValue('website_name'),
    ];
    $form['global_settings']['settings_dialog_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Settings dialog label'),
      '#description' => $this->t('Displayed next to the website name'),
      '#default_value' => $this->getSettingValue('settings_dialog_label'),
    ];

    $form['global_settings']['reload_on_save'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Reload page on changes in cookie preferences.'),
      '#required' => FALSE,
      '#default_value' => $this->getSettingValue('reload_on_save'),
    ];



    $form['cookiebar'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Cookiebar'),
    ];

    $form['cookiebar']['message'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#cols' => 80,
      '#title' => $this->t('Consent message'),
      '#required' => TRUE,
      '#default_value' => $this->getSettingValue('message'),
    ];

    $form['cookiebar']['enable_deny_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable the deny button'),
      '#default_value' => $this->getSettingValue('enable_deny_button'),
      '#description' => $this->t('Give a user the explicit option to deny all cookies except functional cookies.'),
    ];

    $form['cookiebar']['enable_close_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable the close button'),
      '#default_value' => $this->getSettingValue('enable_close_button'),
      '#description' => $this->t('The close button (x) is not needed if a user has the ability to reject or accept cookies.'),
    ];

    $form['cookiebar']['close_button_accept'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Accept on close button'),
      '#default_value' => $this->getSettingValue('close_button_accept'),
      '#description' => $this->t('Pressing the close button (x) implies that a user accepted the cookies.'),
    ];


    $form['cookiebar']['settings_button_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Settings button label'),
      '#default_value' => $this->getSettingValue('settings_button_label'),
    ];

    $form['cookiebar']['accept_button_label'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Accept button label'),
      '#default_value' => $this->getSettingValue('accept_button_label'),
    ];

    $form['cookiebar']['deny_button_label'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Reject button label'),
      '#default_value' => $this->getSettingValue('deny_button_label'),
    ];



    $cookieTypes = ['tracking', 'analytical', 'functional'];

    foreach ($cookieTypes as $cookieType) {
      $form[$cookieType . '_wrapper'] = [
        '#type' => 'fieldset',
        '#title' => ucfirst($cookieType) . ' tab',
      ];

      $form[$cookieType . '_wrapper'][$cookieType . '_title'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Title of %type tab', ['%type' => ucfirst($cookieType)]),
        '#default_value' => $this->getSettingValue($cookieType . '_title'),
      ];
      $form[$cookieType . '_wrapper'][$cookieType . '_description'] = [
        '#type' => 'textarea',
        '#required' => TRUE,
        '#cols' => 80,
        '#rows' => 3,
        '#title' => $this->t('Description of %type', ['%type' => ucfirst($cookieType)]),
        '#default_value' => $this->getSettingValue($cookieType . '_description'),
      ];
      if ($cookieType == 'tracking') {
        $form[$cookieType . '_wrapper'][$cookieType . '_disabled_message'] = [
          '#type' => 'textarea',
          '#required' => TRUE,
          '#cols' => 80,
          '#rows' => 3,
          '#title' => $this->t('No tracking message'),
          '#description' => $this->t('This message will be shown when the user has enabled no tracking in their privacy settings'),
          '#default_value' => $this->getSettingValue($cookieType . '_disabled_message'),
        ];
      }

      $form[$cookieType . '_wrapper'][$cookieType . '_default_state'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('On by default'),
        '#default_value' => $this->getSettingValue($cookieType . '_default_state'),
      ];

      $form[$cookieType . '_wrapper'][$cookieType . '_togglable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('A user can disable this cookie type'),
        '#default_value' => $this->getSettingValue($cookieType . '_togglable'),
      ];
      if ($cookieType === 'functional') {
        $form[$cookieType . '_wrapper'][$cookieType . '_togglable']['#disabled'] = true;
        $form[$cookieType . '_wrapper'][$cookieType . '_togglable']['#default_value'] = false;
        $form[$cookieType . '_wrapper'][$cookieType . '_togglable']['#description'] = $this->t('Functional cookies can never be disabled by a user. They are needed for a site to work properly.');
        $form[$cookieType . '_wrapper'][$cookieType . '_default_state']['#disabled'] = true;
        $form[$cookieType . '_wrapper'][$cookieType . '_default_state']['#default_value'] = true;

      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $gtmCode = $form_state->getValue('gtm_code');
    if (!preg_match('/^GTM-([A-Z0-9]{6,12})$/i', $gtmCode)) {
      $form_state->setErrorByName('gtm_code', $this->t('Invalid GTM code.'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $settings = $this->config(static::settings);
    $formValues = $form_state->getValues();
    $cleanKeys = array_merge($form_state->getCleanValueKeys(),['submit']);
    foreach ($formValues as $settingsKey => $settingsValue) {
      if (!in_array($settingsKey, $cleanKeys, TRUE)) {
        $settings->set($settingsKey, $settingsValue);
      }
    }
    $settings->save();
    parent::submitForm($form, $form_state);
  }

  private function getSettingValue($key) {
    $settings = $this->config(static::settings);
    $defaultSettings = $this->defaultConfigValues();
    return $settings->get($key) ?? $defaultSettings[$key];
  }

}
